﻿using System;
using System.IO;
using System.Text;
using System.Linq;
using System.IO.Ports;
using System.Collections.Generic;

namespace PalmSens
{
    /// <summary>
    /// Command line tool for programming firmware or settings on the EmStat Pico
    /// </summary>
    public static class EsPicoProgrammer
    {
        const string HELP_TXT =
            "EmStat Pico programmer. Used to program firmware, settings or licences.\r\n" +
            "Commands:\r\n" +
            "-help: Display help\r\n" +
            "   No Parameters\r\n" +
            "-upload_fw: Upload an encrypted file\r\n" +
            "   Par1: Input file path\r\n" +
            "   Par2: COM port to upload to\r\n" +
            "-get_uuid: Fetch UUID from device\r\n" +
            "   Par1: COM port\r\n" +
            "-lic_upload: Upload a licence file\r\n" +
            "   Par1: Licence file path\r\n" +
            "   Par2: COM port to upload to\r\n" +
            "-config: Program settings stored in flash\r\n" +
            "   Par1: Settings bits\r\n" +
            "   Par2: COM port to upload to\r\n";

        public static int Main(string[] args)
        {
            try
            {
                //Help is default cmd
                string cmd = args.Length == 0 ? "-help" : args[0];
                switch (cmd)
                {
                    case "-help":
                        Console.WriteLine(HELP_TXT);
                        break;
                    case "-upload_fw":
                        {
                            if (args.Length < 3)
                                throw new ArgumentException("Invalid number of arguments, use -help command for more info");
                            Console.WriteLine("Uploading FW");
                            UploadFirmware(args[1], args[2]);
                            Console.WriteLine("Uploading FW successful");
                            break;
                        }
                    case "-get_uuid":
                        {
                            if (args.Length < 2)
                                throw new ArgumentException("Invalid number of arguments, use -help command for more info");
                            Console.WriteLine("Fetching UUID");
                            GetUUID(args[1]);
                            Console.WriteLine("Fetching UUID successful");
                            break;
                        }
                    case "-lic_upload":
                        {
                            if (args.Length < 3)
                                throw new ArgumentException("Invalid number of arguments, use -help command for more info");
                            Console.WriteLine("Uploading licence");
                            UploadLicence(args[1], args[2]);
                            UploadConfig(0, args[2]); //TODO: temp to make it easier for now
                            Console.WriteLine("Uploading licence successful");
                            break;
                        }
                    case "-config":
                        {
                            if (args.Length < 3)
                                throw new ArgumentException("Invalid number of arguments, use -help command for more info");
                            Console.WriteLine("Uploading licence");
                            UploadConfig(Convert.ToUInt32(args[1]), args[2]);
                            Console.WriteLine("Uploading licence successful");
                            break;
                        }
                    default:
                        throw new ArgumentException("Unknown command");
                }
            }
            catch(Exception ex)
            {
                Console.WriteLine($"Error: Exception thrown: {ex.Message}");
                return 1;
            }
            return 0;
        }

        private static StreamDevice OpenSerial(string comPort)
        {
            SerialPort port = new SerialPort(comPort, 230400, Parity.None, 8, StopBits.One);
            port.ReadBufferSize = 0x1000;
            port.DtrEnable = true;
            port.Open();
            return new StreamDevice(port.BaseStream);
        }

        public static void UploadFirmware(string inFilename, string comPort)
        {
            using (StreamDevice dev = OpenSerial(comPort))
            {
                PSBootloaderEnc bl = new PSBootloaderEnc(dev);
                bl.Init();

                byte[] fwFile = File.ReadAllBytes(inFilename);
                bl.UploadFile(fwFile);
            }
        }
        
        private static void UploadConfig(UInt32 cfg, string comPort)
        {
            using (StreamDevice dev = OpenSerial(comPort))
            {
                string resp = dev.SendCmd("S01" + ToHex(BitConverter.GetBytes(cfg).Reverse().ToArray()));
                if (resp != "\n") throw new IOException("Unexpected answer");
            }
        }

        private static void UploadLicence(string inFilename, string comPort)
        {
            using (StreamDevice dev = OpenSerial(comPort))
            {
                byte[] licFile = File.ReadAllBytes(inFilename);
                string resp = dev.SendCmd("S04" + ToHex(licFile));
                if (resp != "\n") throw new IOException("Unexpected answer");
            }
        }

        private static void GetUUID(string comPort)
        {
            using (StreamDevice dev = OpenSerial(comPort))
            {
                string resp = dev.SendCmd("G05");
                Console.WriteLine(resp.TrimEnd('\n'));
            }
        }

        private static string ToHex(byte[] data)
        {
            StringBuilder str = new StringBuilder(data.Length * 2);
            foreach (byte b in data)
                str.Append(b.ToString("X2"));
            return str.ToString();
        }

        private static byte[] FromHex(string str)
        {
            return Enumerable.Range(0, str.Length / 2).Select(x => Convert.ToByte(str.Substring(x * 2, 2), 16)).ToArray();
        }
    }
}